<?php
declare(strict_types = 1);

namespace PhoxTemplates\Includes\App\Database;

use Exception;
use \WHMCS\Database\Capsule;

/**
 *  Databases
 */
class Setup extends AbstractDatabase
{

    /**
     * Create table phoxtemplate_settings
     *
     * @since 2.0.0
     */
    public function up_db() : array
    {

        //build table
        try{

            $hasTable = $this->schemaBuilder->hasTable(self::settingsTable);

            if (!$hasTable) {
                $this->schemaBuilder->create(self::settingsTable, function ($table){
                    $table->id();
                    $table->string('setting_page_name')->nullable(false)->unique();
                    $table->json('setting_value')->nullable(false);
                    $table->timestamp('updated_at', $precision = 0)->nullable(false);
                });
            }

            $this->init_insert_data_default();

        }catch (Exception $e){

            return [
                'status' => "error",
                'description' => 'Unable to create phoxtemplate_settings: ' . substr($e->getMessage(),0,400),
            ];

        }

        return [
            'status' => 'success',
            'description' => 'The Module Phox Template has been installed successfully'
        ];

    }

    /**
     * Insert default data for all rows
     *
     * @since 2.0.0
     * @return void
     */
    public function init_insert_data_default () : void
    {

        $classes_list = [
            'BrandSettings',
            'StyleSettings',
            'TypoSettings'
        ];

        foreach ( $classes_list as $class_name ){

            $class = __NAMESPACE__ . '\\'. $class_name ;

            $init = new $class();

            call_user_func([$init, 'set_default_value']);

            call_user_func([$init, 'insert_default_value']);

        }


    }

    /**
     * Delete records `template_name` from table `tbladdonmodules` & remove table `phoxtemplate_settings`
     *
     * @since 2.0.0
     * @return array
     */
	public function down_db() : array
	{

        try {

            $this->schemaBuilder->dropIfExists('phoxtemplate_settings');

        }catch (Exception $e){

            return [
                "status" => "error",
                "description" => "Unable to delete record template_name from table phoxtemplate_settings : {$e->getMessage()}"
            ];

        }

        try {

        Capsule::table('tbladdonmodules')->where([
		    ['module', 'phox_templates'],
            ['setting', 'template_name']
        ])->delete();

        }catch (Exception $e){

            return [
                "status" => "error",
                "description" => "Unable to delete record template_name from table tbladdonmodules :" . substr($e->getMessage(),0,400)
            ];

        }

        return [
            'status' => 'success',
            'description' => 'The Module Phox Template has been uninstalled successfully'
        ];

    }


    /**
     * Check if `template_name` setting on table `tbladdonmodules` has been change need to change
     * or that have the same value
     *
     * @param string $templa_name
     * @return bool
     */
    public function isteplateset( string $templa_name){

        $result = false;

        $adddonTable =  Capsule::table(self::addonTable)->where([['module', '=', 'phox_templates'], ['setting', '=', 'template_name']])->first();

        if(empty( $adddonTable ) ){
            $result = true;
        }else{
            $result = ($adddonTable->value === $templa_name) ? false : true ;
        }

        return $result;
    }

    /**
     * Insert new record to table tbladdonmodules
     * or Update the record by add selected template name in table
     *
     * @param string $template_name
     * @return mixed
     * @throws Exception
     *
     */
    public function insertOrUpdateSelectedTemplate( string $template_name){

        if( empty( $template_name ) OR ! $this->isteplateset($template_name) )  return false;

        try {

            Capsule::table(self::addonTable)->updateOrInsert(
                [
                    'module' => 'phox_templates',
                    'setting' => 'template_name'
                ],
                [
                    'module'  => 'phox_templates',
                    'setting' => 'template_name',
                    'value'   => $template_name
                ]
            );

            return true ;

        }catch (Exception $e){

            echo "Unable to insert Or Update tbladdonmodules: {$e->getMessage()}";

        }

    }

    /**
     * get from record template_name get the template name from table tbladdonmodules
     *
     * @return mixed
     */
    public function getInstalledTemplateName(){
        try {

            $data = Capsule::table(self::addonTable)->where([['module', '=', 'phox_templates'], ['setting', '=', 'template_name']])->first();

            if(is_object($data)){
                return $data->value;
            }else{
                return 'classy';
            }

        }catch (Exception $e){

            echo "Unable to insert Or Update tbladdonmodules: {$e->getMessage()}";

        }
    }

    /**
     * Update option on table `phoxtemplatesettings`
     *
     * @return mixed
     */
    public function updateOption($option, $value){

        try{
            Capsule::table('phoxtemplatesettings')->where('option', $option)->update(['value' => $value]);
        }catch (Exception $e){

            echo "Unable to update {$option} on table phoxtemplatesettings : {$e->getMessage()}";

        }


    }

    public function getAllOptions(){
        try{
            return Capsule::table('phoxtemplatesettings')->select('option', 'value')->get();
        }catch (Exception $e){

            echo "Unable to get all options on table phoxtemplatesettings : {$e->getMessage()}";

        }
    }


}